/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCCALLBACKS_H__
#define __VNCCALLBACKS_H__

/**
 * \file vnccallbacks.h
 *
 * This file defines types for pointers to callback functions that may be
 * provided to the VNC Viewer SDK.  You should normally include vncviewersdk.h
 * instead of including this file directly.
 *
 * All of the callbacks in the VNCViewerCallbacks structure are optional. If an
 * application does not need a particular callback, then it should set the
 * appropriate member of VNCViewerCallbacks to NULL when it calls
 * VNCViewerCreate().
 *
 * \section section_callbacks_threading Threading
 *
 * Almost all VNC Viewer callbacks are always invoked by the viewer thread.
 * You must therefore avoid including code in your callbacks that may disrupt
 * the viewer thread's activities.  See \ref section_thread for more
 * information.
 *
 * The only exception to this rule is the VNCViewerLogCallback(), which the SDK
 * may make from any thread. The SDK may also call the VNCViewerLogCallback()
 * before the viewer thread is started. However, the SDK guarantees that no two
 * threads will enter a VNCViewerLogCallback() at the same time, even if there
 * are multiple viewer instances.
 *
 * \see VNCViewerCallbacks, VNCViewerLogCallback, section_thread
 */

#include "vnctypes.h"
#include "vncmirrorlink.h"

/**
 * \brief Called by the SDK to notify of an error.
 *
 * This callback is associated with the VNC Viewer when you call
 * VNCViewerCreate().  When the connection is lost or some other error occurs,
 * the SDK invokes the callback to notify you of the reason.
 *
 * If you explicitly stop a running VNC Viewer with VNCViewerReset(), then the
 * VNCViewerErrorCallback() will be invoked with error code
 * VNCViewerErrorReset.  This allows you to use the same cleanup code for this
 * case as for error cases.
 *
 * The VNC Viewer thread always makes exactly one call to this callback just
 * before it exits.  You can therefore also use this callback alongside
 * VNCViewerThreadStartedCallback() to monitor the lifetime of the VNC Viewer
 * thread.
 *
 * If MirrorLink High Speed Media Link (HSML) is being used, then you should
 * also terminate the associated HSML session when you receive this callback.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param error The error code.
 * \param pReservedForFutureUse Currently unused.
 *
 * \return The return value of this callback is unused by this version of this
 * SDK.  However, for future compatibility, you should return zero.
 *
 * \see VNCViewerError, VNCViewerCreate, VNCViewerCallbacks,
 * VNCViewerSetContext
 */
typedef vnc_int32_t VNCCALL
VNCViewerErrorCallback(VNCViewer *pViewer,
                       void *pContext,
                       VNCViewerError error,
                       void *pReservedForFutureUse);


/**
 * \brief Called by the Viewer SDK immediately after the VNC Viewer thread has
 * started.
 * 
 * This callback is guaranteed to be the first callback from the Viewer thread
 * and is is primarily intended for use when implementing langauge
 * bindings (such as JNI) which require native threads to be registered.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * 
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCViewerSetContext
 */
typedef void VNCCALL
VNCViewerThreadStartedCallback(VNCViewer *pViewer,
                               void *pContext);

/**
 * \brief Called by the SDK when the VNC server challenges the VNC Viewer for a
 * username and password.
 * 
 * This callback is invoked by the Viewer SDK when the VNC server challenges
 * the VNC Viewer for a username and password.  This takes place after the
 * connection to the server is established but before the VNC protocol
 * 'handshake' is complete.
 *
 * Your application must notify the Viewer SDK of the results by calling
 * VNCViewerAsynchronousCredentialsResult().  This call may be made from within
 * your VNCViewerAsynchronousCredentialsCallback, or at a later time, and from
 * any thread.  This gives your application the flexibility to display UI or
 * fetch the credentials from elsewhere without blocking the viewer thread.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param usernameRequired If usernameRequired is non-zero, the VNC Server
 * requires a username for this session.  If usernameRequired is zero, then the
 * security type selected for the session does not require a username.
 * \param passwordRequired If passwordRequired is non-zero, the VNC Server
 * requires a password for this session.  If passwordRequired is zero, then the
 * security type selected for the session does not require a password.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCViewerSetContext,
 * VNCViewerAsynchronousCredentialsResult
 */
typedef void VNCCALL
VNCViewerAsynchronousCredentialsCallback(VNCViewer *pViewer,
                                         void *pContext,
                                         vnc_int32_t usernameRequired,
                                         vnc_int32_t passwordRequired);

/**
 * \brief Called by the SDK to inform you that the RFB session with the VNC
 * server has been successfully authenticated and is now completely
 * established.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param width The initial width of the framebuffer.
 * \param height The initial height of the framebuffer.
 * \param desktopName The name of the server desktop, as a NUL-terminated UTF-8
 * encoded string.
 * \param pServerNativePixelFormat The native pixel format of the server.
 * Unless you override it by calling VNCViewerSetPixelFormat(), this will be
 * the pixel format that is used for the RFB session.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCPixelFormat,
 * VNCViewerSetPixelFormat
 */
typedef void VNCCALL
VNCViewerServerInitCallback(VNCViewer *pViewer,
                            void *pContext,
                            vnc_uint16_t width,
                            vnc_uint16_t height,
                            const char *desktopname,
                            const VNCPixelFormat *pServerNativePixelFormat);

/**
 * \brief Called by the SDK to instruct the application to lock a section of
 * the framebuffer for exclusive access and to pin it in memory.
 *
 * Updates to the server display are copied into the application's framebuffer
 * by the viewer thread.  To ensure thread-safety, your application must
 * prevent other threads (e.g. the one that is rendering the framebuffer on the
 * local display) from accessing at least the requested section of the
 * framebuffer when the VNCViewerLockRectangleExCallback() is called.
 *
 * Additionally, if the framebuffer is managed by a garbage collector, the
 * application must prevent the garbage collector from moving the framebuffer
 * in memory while a section of it is locked.  This process is often called
 * 'pinning'.
 *
 * Your implementation of this callback should block until the framebuffer can
 * be locked.  When the SDK has finished updating the rectangle, it will call
 * your VNCViewerUnlockRectangleExCallback().
 *
 * The SDK will never attempt to lock more than one rectangle for the same VNC
 * Viewer object at the same time.
 *
 * If the application is unable to lock a rectangle, it should return NULL.  In
 * this case, the update will be discarded, and there will be no corresponding
 * call to the VNCViewerUnlockRectangleExCallback().
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pRectangle The rectangle that should be locked.
 * \param pStride On successful return, this should be filled in with the
 * 'stride' of the application's framebuffer.  This is the difference in bytes
 * between the addresses of the leftmost pixel in consecutive rows of the
 * framebuffer, and is usually the width of the framebuffer in pixels
 * multiplied by the number of bytes required for each pixel.
 * \param reason The reason why the rectangle was locked.
 *
 * \return The address within the framebuffer of the top-left pixel of the
 * locked rectangle.  Return NULL if the rectangle cannot be locked.
 * 
 * \see VNCViewerCreate, VNCViewerUnlockRectangleExCallback,
 * VNCViewerUnlockRectangleExCallback, VNCViewerFrameBufferUpdateStartCallback,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
typedef vnc_uint8_t *VNCCALL
VNCViewerLockRectangleExCallback(VNCViewer *pViewer,
                                 void *pContext,
                                 const VNCRectangle *pRectangle,
                                 size_t *pStride,
                                 VNCViewerLockRectangleReason reason);

/**
 * \brief Called by the SDK to instruct the application to unlock a
 * previously-locked section of the framebuffer to unpin it from memory.
 *
 * The rectangle that is passed to the VNCViewerUnlockRectangleExCallback() is
 * always the last one that was successfully locked with the
 * VNCViewerLockRectangleExCallback().
 * 
 * There is no call to the VNCViewerUnlockRectangleExCallback() corresponding
 * to calls to the VNCViewerLockRectangleExCallback() that return NULL.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pRectangle The rectangle that should be unlocked.  This is always the
 * last rectangle that was successfully locked.
 * \param pLockedData The address that was returned by
 * VNCViewerLockRectangleExCallback() when the rectangle was locked.
 * \param reason The reason why the rectangle was locked.
 *
 * \see VNCViewerCreate, VNCViewerLockRectangleExCallback,
 * VNCViewerFrameBufferUpdateStartCallback,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
typedef void VNCCALL
VNCViewerUnlockRectangleExCallback(VNCViewer *pViewer,
                                   void *pContext,
                                   const VNCRectangle *pRectangle,
                                   vnc_uint8_t *pLockedData,
                                   VNCViewerLockRectangleReason reason);

/**
 * \brief Called by the SDK to notify you that the size of the VNC Server's
 * display has changed.
 *
 * When you receive this notification, you should allocate a new framebuffer to
 * store your local copy of the server display.  The old framebuffer should be
 * destroyed. If you are using MirrorLink HSML, then you should change the
 * framebuffer used for HSML pixel data instead of that being used with the
 * viewer SDK.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param width The new width of the server display.
 * \param height The new height of the server display.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks
 */
typedef void VNCCALL
VNCViewerDesktopResizeCallback(VNCViewer *pViewer,
                               void *pContext,
                               vnc_uint16_t width,
                               vnc_uint16_t height);

/**
 * \brief Called by the SDK to allow logging from the SDK's internals.
 *
 * The information logged by this callback is intended for use by application
 * developers and RealVNC support engineers.  It is not localized.
 *
 * Use this callback in conjunction with VNCParameterLog to select and gather
 * logging information from the SDK.
 *
 * \note The SDK may invoke this callback from any thread, including calls made
 * before the viewer thread is started. However, the SDK guarantees that no two
 * threads will enter a VNCViewerLogCallback() at the same time, even if there
 * are multiple viewer instances.
 *
 * \note It is common for VNCViewerLogCallback implementations to call
 * VNCViewerGetContext() (e.g. to retrieve a file handle to which to write the
 * message). If this is the case in your application, then you should ensure
 * that you call VNCViewerSetContext() immediately after VNCViewerCreate() and
 * before calling any other SDK API.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param category The category of the log message, as a NUL-terminated UTF-8
 * encoded string.
 * \param severity The severity of the log message (see VNCParameterLog for an
 * explanation).
 * \param text The text of the log message, as a NUL-terminated UTF-8 encoded
 * string.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCParameterLog
 */
typedef void VNCCALL
VNCViewerLogCallback(VNCViewer *pViewer,
                     void *pContext,
                     const char *category,
                     vnc_int32_t severity,
                     const char *text);

/**
 * \brief Called by the SDK to validate a password that has been supplied by a
 * VNC Server.
 *
 * When you receive this notification, you should validate that the password is
 * one that you recognize.  This is used by the RA2Rev and RA2Revne security
 * types.
 *
 * Your application must notify the Viewer SDK of the results by calling
 * VNCViewerAsynchronousValidatePasswordResult().  This call may be made from
 * within your VNCViewerAsynchronousValidatePasswordCallback, or at a later
 * time, and from any thread.  This gives your application the flexibility to
 * display UI or validate the password externally without blocking the viewer
 * thread.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param password The password supplied by the VNC Server.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks,
 * VNCViewerAsynchronousValidatePasswordCallback,
 * VNCViewerAsynchronousValidatePasswordResult
 */
typedef void VNCCALL
VNCViewerAsynchronousValidatePasswordCallback(VNCViewer *pViewer,
                                              void *pContext,
                                              const char* password);

/**
 * \brief Called by the SDK to inform the application of progress while
 * establishing a VNC session.
 *
 * The viewer thread may invoke this callback once or more while attempting to
 * establish a VNC session.  Your application may use the notification to
 * inform the end user of progress, or to assist in diagnosing connection
 * difficulties.
 *
 * When a VNC session is fully established, the viewer thread will notify
 * VNCViewerSessionProgressSessionEstablished via the
 * VNCViewerSessionProgressCallback().  There are no further calls to the
 * VNCViewerSessionProgressCallback() after this point.
 *
 * When a VNC session cannot be established, or when an established session is
 * lost or terminated, the viewer thread will invoke the VNCViewerErrorCallback
 * and then exit.  There are no calls to the VNCViewerSessionProgressCallback
 * at or after this point.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param sessionProgress Indicates the next action that will be taken by the
 * viewer thread in attempting to establish the VNC session.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCViewerSessionProgress,
 * VNCViewerServerInitCallback, VNCViewerErrorCallback
 */
typedef void VNCCALL
VNCViewerSessionProgressCallback(VNCViewer *pViewer,
                                 void *pContext,
                                 VNCViewerSessionProgress sessionProgress);

/**
 * \brief Called by the SDK to inform the application that a skinface has been
 * selected.
 *
 * If a mobile device skin is in use by a VNC session, then the Viewer SDK will
 * decide which skinface to display based on the current size and orientation
 * of the server display.  The SDK re-makes this decision whenever the size or
 * orientation of the server display change, or if the VNC Server indicates
 * that a slide or flip has opened or closed.
 *
 * This callback notifies your application that the SDK has made a new decision 
 * regarding the choice of skinface.  The result may be the same as the last
 * decision that the SDK made.
 *
 * This callback may be helpful when attempting to debug mobile device skin
 * behaviour.  It also serves as notification that the SDK has recalculated the
 * server desktop display rectangle, which can be useful to know if your
 * application is logging the contents of the server display or has some other
 * reason to want to know where it is.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param skinName The skin name, as defined by the skin XML, or NULL if no
 * skin is being used.
 * \param skinFaceId The skinface ID, as defined by the skin XML, or -1 if no
 * skinface is being used.
 * \param skinFaceName The skinface name, as defined by the skin XML, or NULL
 * if no skinface is being used.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCViewerSetSkin
 */
typedef void VNCCALL
VNCViewerSkinFaceDeterminedCallback(VNCViewer *pViewer,
                                    void *pViewerContext,
                                    const char *skinName,
                                    vnc_int32_t skinFaceId,
                                    const char *skinFaceName);

/**
 * \brief Called by the SDK to notify your application that it should begin
 * fetching a mobile device skin.
 * 
 * The Viewer SDK provides the facility for your application to decide after
 * connecting to a VNC Server whether it is appropriate to use a mobile device
 * skin for a particular server host, and if so, which skin should be used.
 * The runtime properties VNCViewerPropertyServerHostManufacturer,
 * VNCViewerPropertyServerHostModel and VNCViewerPropertyServerHostVersion are
 * designed to aid your application in this decision.
 *
 * To make use of this facility, you should register one or more
 * VNCViewerSkinFetchCallback()s after you create your VNC Viewer object.
 * Once the SDK has established a connection to the VNC Server and determined
 * that the VNC Server is hosted on a mobile device, it will invoke each
 * VNCViewerSkinFetchCallback(), in the order that they were registered, until
 * one of the callbacks indicates that skin fetching should stop.
 *
 * It is recommended, though not absolutely essential, that
 * your VNCViewerSkinFetchCallback()s does not block the viewer thread for any
 * significant length of time.  For example, you can create a background thread
 * to perform the skin fetch, or use whatever asynchronous IO APIs are provided
 * by your viewer platform.
 *
 * Regardless of whether you implement the callback synchronously or
 * asynchronously, you MUST make exactly one call to
 * VNCViewerSkinFetchCallbackComplete() for each VNCViewerSkinFetchCallback()
 * that the Viewer SDK invokes.  The parameter to
 * VNCViewerSkinFetchCallbackComplete() indicates to the Viewer SDK whether it
 * should continue to invoke the remainder of the
 * VNCViewerSkinFetchCallback()s, or whether it should stop.
 *
 * VNCViewerSkinFetchCallback()s are registered with
 * VNCViewerAddSkinFetchCallback() and remain registered for the lifetime of
 * the VNC Viewer object or until you call VNCViewerRemoveSkinFetchCallback()
 * or VNCViewerRemoveAllSkinFetchCallbacks().
 *
 * \section VNCViewerSkinFetchCallback_example Example
 *
 * \code
 * extern VNCViewerSDK sdk;
 *
 * // Returns TRUE if the skin filename was successfully determined.
 * int obtainSkinForDevice(const char *manufacturer,
 *                         const char *model,
 *                         const char **pSkinFilename);
 *
 * // The SDK will invoke this callback after the handshake is complete but
 * // before invoking the VNCViewerServerInitCallback().
 * static void fetchSkin(VNCViewer *pViewer,
 *                       void *pViewerContext,
 *                       void *pSkinFetchCallbackContext)
 * {
 *   char *manufacturer = (*sdk.vncViewerGetPropertyString)(
 *           pViewer, VNCViewerPropertyServerHostManufacturer);
 *   char *model = (*sdk.vncViewerGetPropertyString)(
 *           pViewer, VNCViewerPropertyServerHostModel);
 *
 *   if (manufacturer && model)
 *   {
 *     const char *skinFilename = NULL;
 *
 *     if (obtainSkinForDevice(manufacturer, model, &skinFilename))
 *     {
 *       // We have found a skin for this device.  Apply it to the session and
 *       // tell the SDK not to invoke any more callbacks.
 *       (*sdk.vncViewerSetSkin)(pViewer, skinFilename, 0);
 *       (*sdk.vncViewerSkinFetchCallbackComplete)(pViewer, TRUE);
 *     }
 *     else
 *     {
 *       // We could not find a skin for this device.  Tell the SDK that it
 *       // should continue with the next callback that was registered.
 *       (*sdk.vncViewerSkinFetchCallbackComplete)(pViewer, FALSE);
 *     }
 *   }
 *
 *   (*sdk.vncFreeString)(manufacturer);
 *   (*sdk.vncFreeString)(model);
 * }
 * \endcode
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pViewerContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pSkinFetchCallbackContext The context pointer associated with this
 * VNCViewerSkinFetchCallback() by VNCViewerAddSkinFetchCallback().  This
 * allows your application to use the same VNCViewerSkinFetchCallback() with
 * multiple different configurations (for example, to attempt a download from
 * each of several HTTP servers in turn).
 * 
 * \see VNCViewerSkinFetchCallbackComplete(), 
 * VNCViewerAddSkinFetchCallback(), VNCViewerRemoveSkinFetchCallback(),
 * VNCViewerRemoveAllSkinFetchCallbacks()
 */
typedef void VNCCALL
VNCViewerSkinFetchCallback(VNCViewer *pViewer,
                           void *pViewerContext,
                           void *pSkinFetchCallbackContext);

/**
 * \brief Called by the SDK to inform the application that the VNC server
 * host's clipboard contents have been updated.
 *
 * You can use this notification to update the local clipboard contents so that
 * they match the server's.  This allows the user to copy text from the server'
 * clipboard to a local application.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param clipboardText The text that has been copied to the server's
 * clipboard.  This text is UTF-8 encoded and is NOT NUL-terminated.  It is
 * owned by the SDK, which may overwrite it or free it after
 * VNCViewerServerCutTextCallback() returns.
 * \param clipboardTextLength The length of clipboardText, in bytes.
 *
 * \see VNCViewerCreate, VNCViewerCallbacks, VNCViewerSendClientCutText
 */
typedef void VNCCALL
VNCViewerServerCutTextCallback(VNCViewer *pViewer,
                               void *pContext,
                               const char *clipboardText,
                               size_t clipboardTextLength);

/**
 * \brief Notifies the application that the VNC Server wants to ring the local
 * bell.
 *
 * This callback is invoked when the viewer object receives an RFB Bell message
 * from the VNC Server. The application may respond to this by passing the
 * notification along to the user. For example:
 *
 *  -# by sounding the system beep
 *  -# by playing an alert from the system sound palette
 *  -# by flashing or otherwise updating the display
 *
 * As with other callbacks, this callback is optional. If no action for Bell
 * messages is required, then it is always safe not to provide this callback.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 *
 * \see VNCViewerCreate, VNCViewerCallbacks
 */
typedef void VNCCALL 
VNCViewerBellCallback(VNCViewer *pViewer,
                      void *pContext);

/**
 * \brief Called by the SDK to inform the application that a series of
 * framebuffer updates is about to begin.
 *
 * Each RFB 4 framebuffer update sent by a VNC Server consists of:
 *
 * -# a single FramebufferUpdateStart message
 * -# zero or more Rectangle messages
 * -# a single FramebufferUpdateEnd message
 * 
 * On receipt of a FramebufferUpdateStart message, the SDK invokes this
 * callback.
 *
 * The server uses the FramebufferUpdateStart and FramebufferUpdateEnd messages
 * to group updates into sets that it considers to represent atomic updates to
 * the framebuffer.  If this information is useful to your application, you
 * should implement this callback.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 *
 * \see VNCViewerFrameBufferUpdateEndExCallback,
 * VNCViewerFrameBufferUpdateRequest, VNCViewerEnableDisplayUpdates
 */
typedef void VNCCALL
VNCViewerFrameBufferUpdateStartCallback(VNCViewer *pViewer,
                                        void *pContext);

/**
 * \brief Called by the SDK to inform the application that a series of
 * framebuffer updates has finished. Gives information about the udpated
 * regions.
 *
 * Each RFB 4 framebuffer update sent by a VNC Server consists of:
 *
 * -# a single FramebufferUpdateStart message
 * -# zero or more Rectangle messages
 * -# a single FramebufferUpdateEnd message
 * 
 * On receipt of a FramebufferUpdateEnd message, the SDK invokes this
 * callback.
 *
 * The server uses the FramebufferUpdateStart and FramebufferUpdateEnd messages
 * to group updates into sets that it considers to represent atomic updates to
 * the framebuffer.  If this information is useful to your application, you
 * should implement this callback.
 *
 * The callback specifies which rectangles have been invalidated by the update.
 * This aids the application in deciding what region to redraw. The returned
 * rectangles depend on the SDK mode of operation, which is set through
 * VNCViewerSetParameter, for the parameter VNCParameterInvalidRegionTracking.
 * For each mode the returned rectangles are:
 *
 * -# For no region tracking the rectangle array is NULL and the count is 0
 * -# For tracking a bounding rectangle the array contains one element which is
 * the rectangle that surrounds the invalidated region. The count is
 * always 1.
 * -# For full tracking the array contains the invalidated region
 * as rectangles and the count reflects the number of rectangles.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pInvalidatedRectangles The array of invalidated regions. Its contents
 * depend on the mode of operation.
 * \param rectangleCount The number of regions in the array.
 *
 * \see VNCViewerFrameBufferUpdateStartCallback,
 * VNCViewerFrameBufferUpdateRequest, VNCViewerEnableDisplayUpdates,
 * VNCParameterInvalidRegionTracking
 */
typedef void VNCCALL
VNCViewerFrameBufferUpdateEndExCallback(VNCViewer *pViewer,
                                void *pContext,
                                const VNCRectangle *pInvalidatedRectangles,
                                size_t rectangleCount);

/**
 * \brief Called by the SDK to give the application the opportunity to verify
 * the server by examining its public key.
 *
 * During the security negotiation for RA2-based security types, the viewer and
 * server exchange RSA public keys.  If the server public key is known by the
 * application or the viewer user in advance, then the application may choose
 * to verify the received server public key against the expected value.  (The
 * SDK always challenges the server private key as part of the RA2 negotiation;
 * there is no API corresponding to this step.)
 *
 * The runtime property VNCViewerPropertyServerPublicKey can be used to
 * retrieve the server public key.  The runtime property
 * VNCViewerPropertyServerSignature returns a prefix of a hash of the server
 * public key, and is useful for display to end users.  (The same signature is
 * displayed by VNC Server applications, allowing the user to verify the server
 * identity visually on their first connection to that server.)
 *
 * Your application should call
 * VNCViewerAsynchronousVerifyServerIdentityResult() to notify the SDK of the
 * result of the verification.  This can be done from within the
 * VNCViewerAsynchronousVerifyServerIdentityCallback or at a later time and
 * from another thread.
 *
 * If your application does not provide a
 * VNCViewerAsynchronousVerifyServerIdentityCallback, then all server identity
 * verification is considered to succeed implicitly.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 *
 * \see VNCSecurityType, VNCViewerPropertyServerPublicKey,
 * VNCViewerPropertyServerSignature,
 * VNCViewerAsynchronousVerifyServerIdentityResult()
 */
typedef void VNCCALL
VNCViewerAsynchronousVerifyServerIdentityCallback(VNCViewer *pViewer,
                                                  void *pContext);

/**
 * \brief Provides performance metric results for a particular sample period.
 *
 * Results are only provided if performance measuring has been enabled by the
 * \ref VNCParameterPerformanceMeasuring parameter.
 *
 * The results are provided as a UTF-8 string holding a series of
 * comma-separated values. A header is also written when performance monitoring
 * commences.  This header is prefixed with a '#' character to allow separation
 * from result lines. Each comma-separated element in the string shall
 * constitute a column.
 *
 * A column shall be provided for each metric in the string, including metrics 
 * whose measuring has been disabled. An empty column indicates that the 
 * corresponding metric has been disabled. If a metric is immeasurable for the
 * entire sample period, the text "#Undef" shall be given instead of a value.
 * 
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param text The text of the header or result line being provided, as a NULL-
 * terminated UTF-8 encoded string.
 *
 * \see VNCParameterPerformanceMeasuring
 */
typedef void VNCCALL 
VNCViewerPerformanceResultCallback(VNCViewer *pViewer,
                                   void *pContext,
                                   const char *text);

/**
 * \brief Notifies the application that a remote feature check has succeeded.
 * 
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param featureCheckID The feature check ID, as returned by
 * VNCViewerAddRemoteFeatureCheck().
 * \param featureID Identifies the feature for which the VNC Server was able to
 * answer the challenge. This is useful in the case where the call to
 * VNCViewerAddRemoteFeatureCheck() allowed a choice of features.
 *
 * \see VNCViewerAddRemoteFeatureCheck()
 */
typedef void VNCCALL 
VNCViewerRemoteFeatureCheckSucceededCallback(VNCViewer *pViewer,
                                             void *pContext,
                                             vnc_uint32_t featureCheckID,
                                             vnc_uint32_t featureID);

/**
 * \brief Notifies the application that a remote feature check has failed.
 *
 * The application has the choice to allow the session to continue (perhaps
 * with restricted functionality) or to instruct the SDK to terminate the
 * session. If the session is to be terminated, then the error reported by the
 * VNCViewerErrorCallback() will be VNCViewerErrorNotLicensedForServer.
 * 
 * As with other callbacks, this callback is optional. If no
 * VNCViewerRemoteFeatureCheckSucceededCallback() is provided, then
 * the SDK will always terminate the session.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param featureCheckID The feature check ID, as returned by
 * VNCViewerAddRemoteFeatureCheck().
 *
 * \return Return non-zero if the SDK should terminate the session, or zero if
 * the SDK should allow the session to continue.
 *
 * \see VNCViewerAddRemoteFeatureCheck()
 */
typedef vnc_int32_t VNCCALL 
VNCViewerRemoteFeatureCheckFailedCallback(VNCViewer *pViewer,
                                          void *pContext,
                                          vnc_uint32_t featureCheckID);

/**
 * \brief Notifies the application that the cursor has been updated.
 *
 * \warning This functionality is work-in-progress and is not ready for
 * production use. Your application should not provide this callback.
 */
typedef void VNCCALL
VNCViewerCursorUpdateCallback(VNCViewer *pViewer,
                              void *pContext,
                              const VNCCursor *cursor);

/**
 * \brief Notifies the application of the location of the remote framebuffer in
 * the pixel data provided by the SDK, which can contain other visual data such
 * as skins.
 *
 * The new location will be in relation to the last desktop resize callback
 * that the viewer SDK has issued, so in cases where skins are enabled the
 * desktop resize callback should occur before the remote framebuffer moved
 * callback.
 *
 * If skins aren’t being used then the remote framebuffer moved callback will
 * still be issued, but it will always be a rectangle from (0, 0) to (remote
 * width, remote height).
 */
typedef void VNCCALL
VNCViewerRemoteFramebufferMoved(VNCViewer* pViewer,
                                void *pContext,
                                const VNCRectangle* pNewLocation);

/**
 * \brief Lets the application know whether or not a pluggable renderer
 * is loaded.
 *
 * This callback should always fire when a connection is started. The
 * status will not change over the course of a connection.
 *
 * This callback should occur immediately after the thread started
 * callback.
 *
 * If a pluggable renderer is active, before receiving the \ref
 * VNCViewerFramebufferStateChangedCallback callback, the application
 * is free to ignore the contents of the framebuffer, and leave rendering
 * up to the pluggable renderer. The lock/unlock rectangle callbacks should
 * never be invoked while a pluggable renderer is active and the
 * \ref VNCViewerFramebufferStateChangedCallback is never invoked.
 */
typedef void VNCCALL
VNCViewerPluggableRendererLoadCallback(VNCViewer *pViewer,
                                       void *pContext,
                                       vnc_bool_t rendererFactoryLoaded,
                                       vnc_bool_t rendererLoaded);

/**
 * \brief Notifies the application of the device rotation.
 *
 * This callback provides the rotation currently being applied by the
 * server operating system to the display contents. This value is not
 * relevant when interacting with the server through the VNC session,
 * because the VNC session uses a coordinate system that is always in
 * the correct rotation for viewing. It is, however, useful when interacting
 * with the server device on a lower level, for instance using the HID AAP
 * SDK.
 *
 * This callback will be called whenever the server device provides an
 * updated rotation, including at the start of the connection. Not all
 * servers support this functionality.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param rotation The clockwise rotation currently being applied to the
 * display contents, in multiples of 90 degrees.
 */
typedef void VNCCALL
VNCViewerDeviceRotatedCallback(VNCViewer *pViewer,
                               void *pContext,
                               vnc_uint8_t rotation);

/**
 * \brief Inform the application whether or not it is necessary to render the
 * framebuffer.
 *
 * If shouldRenderFramebuffer is set to vnc_true, then the application must
 * render the framebuffer until the next invocation of this callback. Otherwise,
 * the application can choose to stop rendering the framebuffer until the next
 * invocation of this callback. If the application continues rendering the
 * framebuffer while shouldRenderFramebuffer is vnc_false, it is harmless but
 * the rendering will be invisible.
 *
 * The application should default to rendering the framebuffer if this callback
 * is never invoked.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param shouldRenderFramebuffer vnc_true if the viewer application must
 * render the framebuffer. Otherwise, it should be vnc_false.
 */
typedef void VNCCALL
VNCViewerFramebufferStateChangedCallback(VNCViewer *pViewer,
                                         void *pContext,
                                         vnc_bool_t shouldRenderFramebuffer);

/**
 * \brief Structure that holds pointers to callbacks invoked by the SDK.
 *
 * Any of the pointers in this structure may be NULL if you choose not to
 * implement a particular callback.  However, you will almost certainly want to
 * provide a VNCViewerErrorCallback(), because otherwise it will be difficult
 * to terminate your application safely.
 *
 * For a description of each of the callbacks, please see the documentation for
 * the corresponding typedef.
 *
 * \see VNCViewerCreate()
 */
typedef struct
{
    /**
     * \brief Called by the SDK to notify of an error.
     */
    VNCViewerErrorCallback *pErrorCallback;

    /**
     * \brief Called by the SDK to inform you that the server display
     * dimensions and pixel format have been negotiated.
     */
    VNCViewerServerInitCallback *pServerInitCallback;

    /**
     * \brief Called by the SDK to notify you that the size of the VNC Server's
     * display has changed.
     */
    VNCViewerDesktopResizeCallback *pDesktopResizeCallback;

    /**
     * \brief Called by the SDK to allow logging from the SDK's internals.
     */
    VNCViewerLogCallback *pLogCallback;

    /**
     * \brief Called by the SDK to inform the application of progress while
     * establishing a VNC session.
     */
    VNCViewerSessionProgressCallback *pSessionProgressCallback;

    /**
     * \brief Called by the SDK to inform the application that a skinface has
     * been selected.
     */
    VNCViewerSkinFaceDeterminedCallback *pSkinFaceDeterminedCallback;

    /**
     * \brief Called by the SDK to inform the application the VNC server host's
     * clipboard contents have been updated.
     */
    VNCViewerServerCutTextCallback *pServerCutTextCallback;

    /**
     * \brief Called by the SDK to inform the application that a series of
     * framebuffer updates is about to begin.
     */
    VNCViewerFrameBufferUpdateStartCallback *pFrameBufferUpdateStartCallback;

    /**
     * \brief Called by the SDK to inform the application that a series of
     * framebuffer updates has finished. Gives information about the udpated
     * regions.
     */
    VNCViewerFrameBufferUpdateEndExCallback *pFrameBufferUpdateEndCallback;

    /**
     * \brief Called by the SDK when the VNC server challenges the VNC Viewer for a
     * username and password, and you want to report the result asynchronously.
     */
    VNCViewerAsynchronousCredentialsCallback *pAsynchronousCredentialsCallback;

    /**
     * \brief Called by the SDK to ask the application to validate a password,
     * and you want to report the result asynchronously.
     */
    VNCViewerAsynchronousValidatePasswordCallback *pAsynchronousValidatePasswordCallback;

    /**
     * \brief Called by the SDK to instruct the application to lock a section of
     * the framebuffer for exclusive access and to pin it in memory.
     */
    VNCViewerLockRectangleExCallback *pLockRectangleExCallback;

    /**
     * \brief Called by the SDK to instruct the application to unlock a
     * previously-locked section of the framebuffer to unpin it from memory.
     */
    VNCViewerUnlockRectangleExCallback *pUnlockRectangleExCallback;
    
    /**
     * \brief Called by the Viewer SDK immediately after the VNC Viewer thread
     * has started.
     */
    VNCViewerThreadStartedCallback *pThreadStartedCallback;

    /**
     * \brief Notifies the application that the SDK has received the MirrorLink
     * server's ServerDisplayConfiguration.
     */
    VNCViewerServerDisplayConfigurationCallback
        *pServerDisplayConfigurationCallback;

    /**
     * \brief Notifies the application that the SDK has received the MirrorLink
     * server's ServerEventConfiguration.
     */
    VNCViewerServerEventConfigurationCallback
        *pServerEventConfigurationCallback;

    /**
     * \brief Notifies the application that the SDK has received a DeviceStatus
     * MirrorLink extension message from the server.
     */
    VNCViewerDeviceStatusCallback *pDeviceStatusCallback;

    /**
     * \brief Notifies the application that the SDK has received a MirrorLink
     * Context Information rectangle from the server.
     */
    VNCViewerContextInformationCallback *pContextInformationCallback;

    /**
     * \brief Called by the SDK to give the application the opportunity to
     * verify the server by examining its public key.
     */
    VNCViewerAsynchronousVerifyServerIdentityCallback
        *pAsynchronousVerifyServerIdentityCallback;

    /**
     * \brief Provides performance metric results for a particular sample
     * period.
     */
    VNCViewerPerformanceResultCallback *pPerformanceResultCallback;
    
    /** Notifies the application that a remote feature check has succeeded. */
    VNCViewerRemoteFeatureCheckSucceededCallback
        *pRemoteFeatureCheckSucceededCallback;

    /** Notifies the application that a remote feature check has failed. */
    VNCViewerRemoteFeatureCheckFailedCallback
        *pRemoteFeatureCheckFailedCallback;

    /** Notifies the application that the cursor has been updated. */
    VNCViewerCursorUpdateCallback *pCursorUpdateCallback;

    /**
     * \brief Notifies the application that the SDK has received a
     * VirtualKeyboardTrigger MirrorLink extension message from the server.
     */
    VNCViewerVirtualKeyboardTriggerCallback *pVirtualKeyboardTriggerCallback;

    /**
     * \brief Notifies the application that the SDK has received an
     * EventMapping MirrorLink extension message from the server.
     */
    VNCViewerEventMappingCallback *pEventMappingCallback;

    /**
     * \brief Notifies the application that the SDK has received a
     * KeyEventListing MirrorLink extension message from the server.
     */
    VNCViewerKeyEventListingCallback *pKeyEventListingCallback;

    /**
     * \brief Notifies the application that the VNC Server wants to ring the
     * local bell.
     */
    VNCViewerBellCallback *pBellCallback;

    /**
     * \brief Notifies the application that MirrorLink content attestation has
     * failed, or that the server has not attested everything that was
     * requested.
     */
    VNCViewerContentAttestationFailureCallback
        *pContentAttestationFailureCallback;

    /**
     * \brief Requests that the application performs a locale
     * dependent conversion to upper case.
     */
    VNCViewerToUpperCallback *pToUpperCallback;

    /**
     * \brief Requests that the application performs a locale
     * dependent conversion to lower case.
     */
    VNCViewerToLowerCallback *pToLowerCallback;

    /**
     * Reports the outcome of calling
     * VNCViewerSendFramebufferBlockingNotificationEx.
     */
    VNCViewerFramebufferBlockingNotificationSentCallback *pFrambufferBlockingNotificationSentCallback;
    /**
     * Notifies the application of the location of the
     * remote framebuffer in the pixel data provided by the SDK.
     */
    VNCViewerRemoteFramebufferMoved *pRemoteFramebufferMoved;
    /**
     * Lets the application know whether or not a pluggable renderer
     * is loaded.
     */
    VNCViewerPluggableRendererLoadCallback *pPluggableRendererLoadCallback;
    /**
     * Notifies the application that the SDK has received a MirrorLink High
     * Speed Media Link (HSML) Information rectangle from the server.
     */
    VNCViewerHSMLInformationCallback *pHSMLInformationCallback;

    /**
     * \brief Notifies the application of the device rotation.
     */
    VNCViewerDeviceRotatedCallback *pDeviceRotatedCallback;

    /**
     * \brief Inform the application whether or not it is necessary to render
     * the framebuffer.
     */
    VNCViewerFramebufferStateChangedCallback *pFramebufferStateChangedCallback;
} VNCViewerCallbacks;

/**
 * \brief Notifies the application that the VNC Server has enabled or disabled
 * an application-defined protocol extension.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pExtension The VNCExtension pointer returned by
 * VNCViewerRegisterExtension().
 * \param pContext The extension context pointer associated with the VNC Viewer
 * by VNCViewerSetExtensionContext().
 * \param enabled Non-zero if the VNC Server has enabled the extension, or zero
 * if the VNC Server has disabled the extension.
 *
 * \see VNCViewerExtensionMessageReceivedCallback(),
 * VNCViewerRegisterExtension(), VNCViewerSendExtensionMessage()
 */
typedef void VNCCALL
VNCViewerExtensionEnabledCallback(VNCViewer *pViewer,
                                  void *pContext,
                                  VNCExtension *pExtension,
                                  void *pExtensionContext,
                                  vnc_int32_t enabled);

/**
 * \brief Notifies the application that an extension message for an
 * application-defined protocol extension has been received.
 *
 * The SDK may free or overwrite the extension message payload as soon as the
 * VNCViewerExtensionMessageReceivedCallback returns.  If the application
 * requires the data to persist for longer than this, then it must copy it.
 * 
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pExtension The VNCExtension pointer returned by
 * VNCViewerRegisterExtension().
 * \param pContext The extension context pointer associated with the VNC Viewer
 * by VNCViewerSetExtensionContext().
 * \param payload The payload of the extension message.  This memory is owned
 * by the SDK, which may overwrite it or free it after
 * VNCViewerExtensionMessageReceivedCallback() returns.
 * \param payloadLength The length of the payload of the extension message.
 *
 * \see VNCViewerExtensionEnabledCallback(), VNCViewerRegisterExtension(),
 * VNCViewerSendExtensionMessage()
 */
typedef void VNCCALL
VNCViewerExtensionMessageReceivedCallback(VNCViewer *pViewer,
                                          void *pContext,
                                          VNCExtension *pExtension,
                                          void *pExtensionContext,
                                          const vnc_uint8_t *payload,
                                          size_t payloadLength);

#endif /* !defined(__VNCCALLBACKS_H__) */
